import React, { useState, useEffect } from 'react';
import LuxuryRsvpForm from './LuxuryRsvpForm';
import SuccessView from './SuccessView';

// Content configuration
const CONTENT = {
    preheader: 'LOOKING FORWARD TO CELEBRATING WITH YOU',
    rsvpTitle: 'R  S  V  P',
};

export default function InvitePage({ token }) {
    // Styles
    const styles = {
        bg: '#2A0713',
        ink: '#C9B699',
        inkSoft: 'rgba(201,182,153,0.8)',
        fieldBorder: '#7B5F51',
        accent: '#B6A37F',
        accentHover: '#C8B588',
        error: '#E8CBB5'
    };
    const [data, setData] = useState(null);
    const [loading, setLoading] = useState(true);
    const [submitted, setSubmitted] = useState(false);
    const [error, setError] = useState(null);

    useEffect(() => {
        fetch(`/api/rsvp/${token}`)
            .then(res => {
                if (!res.ok) {
                    return res.json().then(data => {
                        throw new Error(data.error || `HTTP error! status: ${res.status}`);
                    });
                }
                return res.json();
            })
            .then(data => {
                if (data.error) {
                    setError(data.error);
                } else {
                    setData(data);
                }
                setLoading(false);
            })
            .catch(err => {
                console.error('Failed to load invitation:', err);
                setError(err.message || 'Failed to load invitation');
                setLoading(false);
            });
    }, [token]);

    const handleSubmit = async (formData) => {
        try {
            const res = await fetch(`/api/rsvp/${token}`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(formData)
            });

            const result = await res.json();
            
            if (result.error) {
                alert(result.error);
            } else {
                setSubmitted(true);
            }
        } catch (err) {
            alert('Failed to submit RSVP. Please try again.');
        }
    };

    if (loading) {
        return (
            <div style={{ background: styles.bg }} className="min-h-screen flex items-center justify-center">
                <div style={{ color: styles.ink, fontFamily: 'Inter, sans-serif', letterSpacing: '0.2em' }} className="text-sm uppercase">Loading...</div>
            </div>
        );
    }

    if (error) {
        return (
            <div style={{ background: styles.bg }} className="min-h-screen flex items-center justify-center p-6">
                <div className="text-center" style={{ maxWidth: '560px' }}>
                    <h2 style={{ 
                        fontFamily: "'Playfair Display', serif", 
                        fontSize: 'clamp(1.5rem, 4vw, 2.5rem)', 
                        color: styles.ink,
                        letterSpacing: '0.15em',
                        marginBottom: '24px'
                    }}>
                        OOPS!
                    </h2>
                    <p style={{ 
                        fontFamily: 'Inter, sans-serif', 
                        color: styles.error, 
                        fontSize: '1rem',
                        lineHeight: '1.6'
                    }}>
                        {error}
                    </p>
                </div>
            </div>
        );
    }

    if (submitted) {
        return <SuccessView data={data} />;
    }

    const { mode, household, settings, events } = data;

    return (
        <div style={{ background: styles.bg, minHeight: '100vh', padding: '64px 20px' }}>
            <div style={{ maxWidth: '880px', margin: '0 auto' }}>
                {/* Header */}
                <div style={{ textAlign: 'center', marginBottom: '40px' }}>
                    <p style={{ 
                        fontFamily: 'Inter, sans-serif', 
                        fontSize: '0.75rem', 
                        color: styles.inkSoft, 
                        letterSpacing: '0.15em',
                        textTransform: 'uppercase',
                        marginBottom: '20px'
                    }}>
                        {CONTENT.preheader}
                    </p>
                    
                    <h1 style={{ 
                        fontFamily: "'Bodoni Moda', serif", 
                        fontSize: 'clamp(3rem, 10vw, 6rem)', 
                        color: styles.ink,
                        letterSpacing: '0.3em',
                        lineHeight: '1',
                        marginBottom: '24px',
                        marginTop: '34px',
                        fontWeight: '700'
                    }}>
                        {CONTENT.rsvpTitle}
                    </h1>
                    
                    {events[0]?.nameOne && events[0]?.nameTwo ? (
                        <div style={{ 
                            display: 'grid',
                            gridTemplateColumns: '1fr auto 1fr',
                            alignItems: 'center',
                            marginBottom: '12px',
                            gap: '20px'
                        }}>
                            <div style={{ 
                                textAlign: 'right'
                            }}>
                                <span style={{ 
                                    fontFamily: `'${events[0]?.nameFont || 'Bodoni Moda'}', serif`, 
                                    fontSize: '1.2rem', 
                                    color: styles.ink,
                                    letterSpacing: '0.2em',
                                    textTransform: 'uppercase'
                                }}>
                                    {events[0].nameOne.toUpperCase()}
                                </span>
                            </div>
                            <span style={{ 
                                fontFamily: 'Inter, sans-serif', 
                                fontSize: '0.9rem',
                                color: styles.ink,
                                letterSpacing: '0.1em',
                                whiteSpace: 'nowrap'
                            }}>
                                AND
                            </span>
                            <div style={{ 
                                textAlign: 'left'
                            }}>
                                <span style={{ 
                                    fontFamily: `'${events[0]?.nameFont || 'Bodoni Moda'}', serif`, 
                                    fontSize: '1.2rem', 
                                    color: styles.ink,
                                    letterSpacing: '0.2em',
                                    textTransform: 'uppercase'
                                }}>
                                    {events[0].nameTwo.toUpperCase()}
                                </span>
                            </div>
                        </div>
                    ) : (
                        <p style={{ 
                            fontFamily: `'${events[0]?.nameFont || 'Bodoni Moda'}', serif`, 
                            fontSize: '1.2rem', 
                            color: styles.ink,
                            letterSpacing: '0.2em',
                            textTransform: 'uppercase',
                            marginBottom: '12px'
                        }}>
                            {events[0]?.name?.toUpperCase() || 'WEDDING CEREMONY'}
                        </p>
                    )}
                    
                    {events[0]?.startAt && (
                        <p style={{ 
                            fontFamily: 'Inter, sans-serif', 
                            fontSize: '0.75rem', 
                            color: styles.inkSoft,
                            letterSpacing: '0.1em',
                            textTransform: 'uppercase',
                            marginBottom: '8px'
                        }}>
                            {new Date(events[0].startAt).toLocaleDateString('en-US', {
                                weekday: 'long',
                                year: 'numeric',
                                month: 'long',
                                day: 'numeric'
                            }).toUpperCase()}
                        </p>
                    )}
                    
                    {events[0] && (
                        <p style={{ 
                            fontFamily: 'Inter, sans-serif', 
                            fontSize: '0.75rem', 
                            color: styles.inkSoft,
                            letterSpacing: '0.1em',
                            textTransform: 'uppercase'
                        }}>
                            {events[0].venueName ? `VENUE: ${events[0].venueName.toUpperCase()}` : ''}{events[0].venueName && events[0].startAt ? '. ' : ''}TIME: {events[0].startAt ? new Date(events[0].startAt.replace(' ', 'T')).toLocaleTimeString('en-US', { hour: 'numeric', minute: '2-digit' }).toUpperCase() : ''}{events[0].endAt ? ` - ${new Date(events[0].endAt.replace(' ', 'T')).toLocaleTimeString('en-US', { hour: 'numeric', minute: '2-digit' }).toUpperCase()}` : ''}
                        </p>
                    )}
                </div>

                {/* Welcome Message */}
                <div style={{ 
                    textAlign: 'center', 
                    marginBottom: '40px',
                    padding: '24px',
                    border: `1px solid ${styles.fieldBorder}`,
                    borderRadius: '4px'
                }}>
                    <p style={{ 
                        fontFamily: "'Playfair Display', serif", 
                        fontSize: '1.5rem', 
                        color: styles.ink,
                        letterSpacing: '0.1em'
                    }}>
                        Welcome, {household.displayName}!
                    </p>
                </div>
                
                {settings.rsvpDeadline && new Date() > new Date(settings.rsvpDeadline) ? (
                    <div style={{
                        background: 'rgba(232,203,181,0.1)',
                        border: `1px solid ${styles.error}`,
                        borderRadius: '4px',
                        padding: '24px',
                        textAlign: 'center'
                    }}>
                        <p style={{ 
                            fontFamily: 'Inter, sans-serif',
                            fontSize: '0.875rem',
                            color: styles.error,
                            letterSpacing: '0.1em',
                            textTransform: 'uppercase',
                            marginBottom: '8px',
                            fontWeight: '600'
                        }}>
                            RSVP DEADLINE PASSED
                        </p>
                        <p style={{ 
                            fontFamily: 'Inter, sans-serif',
                            color: styles.inkSoft,
                            fontSize: '0.9rem'
                        }}>
                            The RSVP deadline has passed. Please contact us directly if you need to make changes.
                        </p>
                    </div>
                ) : (
                    <LuxuryRsvpForm 
                        events={events}
                        mode={mode}
                        onSubmit={handleSubmit}
                        styles={styles}
                        household={household}
                        settings={settings}
                    />
                )}
            </div>
        </div>
    );
}
